<?php
/**
 * OXO.GE API — HTTP communication layer
 *
 * @package OxoGE
 * @since   3.0.0
 */

if (!defined('ABSPATH')) exit;

/**
 * GET request to OXO.GE stats API.
 *
 * @param string $action  API action name (summary, daily, hourly, geo, …)
 * @param array  $extra   Additional query params
 * @return array           Decoded JSON response or ['error' => '…']
 */
function oxo_ge_api_get($action, $extra = []) {
    $api_key = get_option('oxo_ge_api_key', '');
    if (empty($api_key)) {
        return ['error' => 'API key არ არის კონფიგურირებული'];
    }

    $params = array_merge(
        ['api_key' => $api_key, 'action' => $action],
        $extra
    );

    $url = OXO_GE_API_BASE . '/site_stats?' . http_build_query($params);
    $res = wp_remote_get($url, ['timeout' => 15, 'sslverify' => false]);

    if (is_wp_error($res)) {
        return ['error' => $res->get_error_message()];
    }

    return json_decode(wp_remote_retrieve_body($res), true)
        ?: ['error' => 'პასუხი ვერ დამუშავდა'];
}

/**
 * POST news to OXO.GE.
 *
 * @param string $title      News title
 * @param string $content    News body
 * @param string $image_url  Optional image URL
 * @return array              Decoded JSON response
 */
function oxo_ge_api_post_news($title, $content, $image_url = '') {
    $api_key = get_option('oxo_ge_api_key', '');
    $site_id = get_option('oxo_ge_site_id', '');

    if (empty($api_key) || empty($site_id)) {
        return ['error' => 'API key ან Site ID არ არის კონფიგურირებული'];
    }

    $body = [
        'site_id' => $site_id,
        'api_key' => $api_key,
        'title'   => $title,
        'content' => $content,
    ];
    if (!empty($image_url)) {
        $body['image_url'] = $image_url;
    }

    $res = wp_remote_post(OXO_GE_API_BASE . '/news_submit', [
        'timeout'   => 15,
        'sslverify' => false,
        'body'      => $body,
    ]);

    if (is_wp_error($res)) {
        return ['error' => $res->get_error_message()];
    }

    return json_decode(wp_remote_retrieve_body($res), true)
        ?: ['error' => 'პასუხი ვერ დამუშავდა'];
}
