<?php
/**
 * OXO.GE Shortcodes — registrations, registry and handlers
 *
 * Available shortcodes:
 *   [oxo-hits]  [oxo-unique]  [oxo-online]  [oxo-rank]  [oxo-georgia]
 *   [oxo-monthly]  [oxo-hits-yesterday]  [oxo-unique-yesterday]
 *   [oxo-avg-daily]  [oxo-male]  [oxo-female]  [oxo-gender]  [oxo-site-name]
 *   [oxo-stats-card stat="oxo-online"]
 *   [oxo-stats-box]
 *
 * @package OxoGE
 * @since   3.0.0
 */

if (!defined('ABSPATH')) exit;

/* ───────────────────────────────────────────
   Registry — single source of truth
   ─────────────────────────────────────────── */

/**
 * Return the full shortcode registry.
 * key  => meta array with label, desc, path, format, icon, example.
 */
function oxo_ge_shortcode_registry() {
    return [
        'oxo-hits' => [
            'label'   => 'ჰიტები დღეს',
            'desc'    => 'დღევანდელი ჰიტების რაოდენობა',
            'path'    => 'today.hits',
            'format'  => 'number',
            'icon'    => '📈',
            'example' => '2,481',
        ],
        'oxo-unique' => [
            'label'   => 'უნიკალური დღეს',
            'desc'    => 'დღევანდელი უნიკალური ვიზიტორები',
            'path'    => 'today.unique_visitors',
            'format'  => 'number',
            'icon'    => '👤',
            'example' => '1,247',
        ],
        'oxo-online' => [
            'label'   => 'ონლაინ ახლა',
            'desc'    => 'ამჟამად საიტზე მყოფი ვიზიტორთა რაოდენობა',
            'path'    => 'today.online_now',
            'format'  => 'int',
            'icon'    => '🟢',
            'example' => '14',
        ],
        'oxo-rank' => [
            'label'   => 'რანკი',
            'desc'    => 'საიტის პოზიცია OXO.GE რეიტინგში',
            'path'    => 'site.rank',
            'format'  => 'rank',
            'icon'    => '🏆',
            'example' => '#21',
        ],
        'oxo-georgia' => [
            'label'   => 'საქართველოდან %',
            'desc'    => 'ვიზიტორთა პროცენტი საქართველოდან',
            'path'    => 'today.georgia_pct',
            'format'  => 'pct',
            'icon'    => '🇬🇪',
            'example' => '87%',
        ],
        'oxo-monthly' => [
            'label'   => 'თვიური უნიკალური',
            'desc'    => 'ყოველთვიური უნიკალური ვიზიტორები (საშუალო)',
            'path'    => 'averages.monthly_uniques',
            'format'  => 'number',
            'icon'    => '📅',
            'example' => '18,420',
        ],
        'oxo-hits-yesterday' => [
            'label'   => 'ჰიტები გუშინ',
            'desc'    => 'გუშინდელი ჰიტების რაოდენობა',
            'path'    => 'yesterday.hits',
            'format'  => 'number',
            'icon'    => '📊',
            'example' => '3,105',
        ],
        'oxo-unique-yesterday' => [
            'label'   => 'უნიკალური გუშინ',
            'desc'    => 'გუშინდელი უნიკალური ვიზიტორები',
            'path'    => 'yesterday.unique_visitors',
            'format'  => 'number',
            'icon'    => '👥',
            'example' => '1,892',
        ],
        'oxo-avg-daily' => [
            'label'   => 'საშუალო დღიური',
            'desc'    => 'საშუალო დღიური ვიზიტორთა რაოდენობა',
            'path'    => 'averages.daily_visitors',
            'format'  => 'number',
            'icon'    => '📉',
            'example' => '956',
        ],
        'oxo-male' => [
            'label'   => 'მამაკაცი %',
            'desc'    => 'მამაკაცი ვიზიტორების პროცენტი',
            'path'    => 'demographics.male_pct',
            'format'  => 'pct',
            'icon'    => '♂',
            'example' => '58%',
        ],
        'oxo-female' => [
            'label'   => 'ქალი %',
            'desc'    => 'ქალი ვიზიტორების პროცენტი',
            'path'    => 'demographics.female_pct',
            'format'  => 'pct',
            'icon'    => '♀',
            'example' => '42%',
        ],
        'oxo-gender' => [
            'label'   => 'სქესი',
            'desc'    => 'მამაკაცი/ქალი პროცენტული თანაფარდობა',
            'path'    => '_gender_combo',
            'format'  => 'gender',
            'icon'    => '⚤',
            'example' => '58% / 42%',
        ],
        'oxo-site-name' => [
            'label'   => 'საიტის სახელი',
            'desc'    => 'საიტის სახელი OXO.GE სისტემაში',
            'path'    => 'site.name',
            'format'  => 'text',
            'icon'    => '🌐',
            'example' => 'example.ge',
        ],
    ];
}

/* ───────────────────────────────────────────
   Value Extraction & Formatting
   ─────────────────────────────────────────── */

/**
 * Walk a dotted path inside a nested array.
 */
function oxo_ge_extract_value($data, $path) {
    foreach (explode('.', $path) as $k) {
        if (!is_array($data) || !isset($data[$k])) return null;
        $data = $data[$k];
    }
    return $data;
}

/**
 * Format a raw value by type.
 */
function oxo_ge_format_value($raw, $format) {
    switch ($format) {
        case 'number': return number_format((int) $raw);
        case 'int':    return (int) $raw;
        case 'rank':   return '#' . (int) $raw;
        case 'pct':    return (int) $raw . '%';
        case 'text':   return esc_html((string) $raw);
        default:       return esc_html((string) $raw);
    }
}

/**
 * Resolve a registry entry to its formatted current value.
 *
 * @param array  $data  Cached summary data
 * @param array  $meta  Registry meta entry
 * @return string|null   Formatted value or null on failure
 */
function oxo_ge_resolve_stat($data, $meta) {
    if ($meta['path'] === '_gender_combo') {
        $m = (int) oxo_ge_extract_value($data, 'demographics.male_pct');
        $f = (int) oxo_ge_extract_value($data, 'demographics.female_pct');
        return $m . '% / ' . $f . '%';
    }

    $raw = oxo_ge_extract_value($data, $meta['path']);
    if ($raw === null) return null;
    return oxo_ge_format_value($raw, $meta['format']);
}

/* ───────────────────────────────────────────
   Shortcode Handlers
   ─────────────────────────────────────────── */

/**
 * Universal handler for all individual stat shortcodes.
 * Attributes: class, style, prefix, suffix, default, icon, label
 */
function oxo_ge_shortcode_handler($atts, $content, $tag) {
    $registry = oxo_ge_shortcode_registry();
    if (!isset($registry[$tag])) return '';
    $meta = $registry[$tag];

    $atts = shortcode_atts([
        'class'   => '',
        'style'   => '',
        'prefix'  => '',
        'suffix'  => '',
        'default' => '—',
        'icon'    => '',
        'label'   => '',
    ], $atts, $tag);

    $data = oxo_ge_get_cached_summary();
    if (!isset($data['success']) || !$data['success']) {
        return '<span class="oxo-stat oxo-stat-error">' . esc_html($atts['default']) . '</span>';
    }

    $value = oxo_ge_resolve_stat($data, $meta);
    if ($value === null) {
        return '<span class="oxo-stat">' . esc_html($atts['default']) . '</span>';
    }

    // CSS classes
    $cls = 'oxo-stat oxo-stat-' . str_replace('oxo-', '', $tag);
    if (!empty($atts['class'])) $cls .= ' ' . sanitize_html_class($atts['class']);

    $style = !empty($atts['style']) ? ' style="' . esc_attr($atts['style']) . '"' : '';

    // Icon
    $icon = '';
    if ($atts['icon'] === 'true' || $atts['icon'] === '1') {
        $icon = '<span class="oxo-stat-icon">' . $meta['icon'] . '</span> ';
    } elseif (!empty($atts['icon']) && $atts['icon'] !== 'false' && $atts['icon'] !== '0') {
        $icon = '<span class="oxo-stat-icon">' . esc_html($atts['icon']) . '</span> ';
    }

    // Label
    $label = '';
    if ($atts['label'] === 'true' || $atts['label'] === '1') {
        $label = ' <span class="oxo-stat-label">' . esc_html($meta['label']) . '</span>';
    } elseif (!empty($atts['label']) && $atts['label'] !== 'false' && $atts['label'] !== '0') {
        $label = ' <span class="oxo-stat-label">' . esc_html($atts['label']) . '</span>';
    }

    $pre = !empty($atts['prefix']) ? esc_html($atts['prefix']) : '';
    $suf = !empty($atts['suffix']) ? esc_html($atts['suffix']) : '';

    return '<span class="' . esc_attr($cls) . '"' . $style . '>'
         . $icon . $pre
         . '<span class="oxo-stat-value">' . $value . '</span>'
         . $suf . $label
         . '</span>';
}

/**
 * [oxo-stats-card stat="oxo-online"]
 * Single stat card with icon, value, label.
 */
function oxo_ge_stats_card_handler($atts) {
    $atts = shortcode_atts([
        'stat'  => 'oxo-hits',
        'class' => '',
        'style' => '',
    ], $atts, 'oxo-stats-card');

    $registry = oxo_ge_shortcode_registry();
    $tag = $atts['stat'];
    if (!isset($registry[$tag])) return '';
    $meta = $registry[$tag];

    $data = oxo_ge_get_cached_summary();
    if (!isset($data['success']) || !$data['success']) return '';

    $value = oxo_ge_resolve_stat($data, $meta);
    if ($value === null) return '';

    $cls   = 'oxo-card-widget';
    if (!empty($atts['class'])) $cls .= ' ' . esc_attr($atts['class']);
    $style = !empty($atts['style']) ? ' style="' . esc_attr($atts['style']) . '"' : '';

    return '<div class="' . $cls . '"' . $style . '>'
         . '<span class="oxo-card-widget-icon">' . $meta['icon'] . '</span>'
         . '<span class="oxo-card-widget-value">' . $value . '</span>'
         . '<span class="oxo-card-widget-label">' . esc_html($meta['label']) . '</span>'
         . '</div>';
}

/**
 * [oxo-stats-box title="..." stats="oxo-hits,oxo-unique,..."]
 * Full stats widget with header, grid, footer.
 */
function oxo_ge_stats_box_handler($atts) {
    $atts = shortcode_atts([
        'class' => '',
        'style' => '',
        'title' => 'OXO.GE სტატისტიკა',
        'stats' => 'oxo-hits,oxo-unique,oxo-online,oxo-rank,oxo-georgia,oxo-monthly',
    ], $atts, 'oxo-stats-box');

    $data = oxo_ge_get_cached_summary();
    if (!isset($data['success']) || !$data['success']) {
        return '<div class="oxo-box-widget">მონაცემები მიუწვდომელია</div>';
    }

    $registry  = oxo_ge_shortcode_registry();
    $stat_keys = array_map('trim', explode(',', $atts['stats']));

    $cls   = 'oxo-box-widget';
    if (!empty($atts['class'])) $cls .= ' ' . esc_attr($atts['class']);
    $style = !empty($atts['style']) ? ' style="' . esc_attr($atts['style']) . '"' : '';

    $html = '<div class="' . $cls . '"' . $style . '>';

    if (!empty($atts['title'])) {
        $html .= '<div class="oxo-box-widget-header">'
               . '<span class="oxo-box-widget-logo">OXO</span>'
               . '<span class="oxo-box-widget-title">' . esc_html($atts['title']) . '</span>'
               . '</div>';
    }

    $html .= '<div class="oxo-box-widget-grid">';
    foreach ($stat_keys as $tag) {
        if (!isset($registry[$tag])) continue;
        $meta  = $registry[$tag];
        $value = oxo_ge_resolve_stat($data, $meta) ?? '—';

        $html .= '<div class="oxo-box-widget-item">'
               . '<span class="oxo-box-item-icon">' . $meta['icon'] . '</span>'
               . '<span class="oxo-box-item-value">' . $value . '</span>'
               . '<span class="oxo-box-item-label">' . esc_html($meta['label']) . '</span>'
               . '</div>';
    }
    $html .= '</div>';

    $html .= '<div class="oxo-box-widget-footer">'
           . '<a href="https://oxo.ge" target="_blank" rel="noopener">OXO.GE</a>'
           . '</div>';
    $html .= '</div>';

    return $html;
}

/* ───────────────────────────────────────────
   Registration (called on 'init')
   ─────────────────────────────────────────── */

add_action('init', function () {
    // Register every individual stat shortcode
    foreach (array_keys(oxo_ge_shortcode_registry()) as $tag) {
        add_shortcode($tag, 'oxo_ge_shortcode_handler');
    }

    // Compound shortcodes
    add_shortcode('oxo-stats-card', 'oxo_ge_stats_card_handler');
    add_shortcode('oxo-stats-box',  'oxo_ge_stats_box_handler');
});
